"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BemChatController = exports.bemchatController = void 0;
const jobQueue_service_1 = require("../../services/jobQueue.service");
const parseRequestBody_1 = require("../../utils/parseRequestBody");
const bemchat_service_1 = require("./bemchat.service");
class BemChatController {
    bemchatService;
    constructor(bemchatService) {
        this.bemchatService = bemchatService;
    }
    createOne = async (c) => {
        const body = (await (0, parseRequestBody_1.parseRequestBody)(c));
        // Validações síncronas (rápidas, < 1 segundo)
        if (!body.name || !body.type || !body.register) {
            c.status(400);
            return c.json({
                message: "Missing required fields: name, type, and register",
            });
        }
        // Validar nome da instância (não pode conter underscore)
        if (body.name.includes("_")) {
            c.status(400);
            return c.json({
                message: "Invalid instance name: underscore (_) is not allowed. Use hyphen (-) instead. Example: 'cliente-teste' instead of 'cliente_teste'",
            });
        }
        // Validar formato do nome (apenas letras minúsculas, números e hífen)
        const namePattern = /^[a-z0-9-]+$/;
        if (!namePattern.test(body.name)) {
            c.status(400);
            return c.json({
                message: "Invalid instance name: only lowercase letters, numbers, and hyphens are allowed",
            });
        }
        // Validar campos obrigatórios do register
        const { id, name, companyName, document, state, city } = body.register;
        if (!id || !name || !companyName || !document || !state || !city) {
            c.status(400);
            return c.json({
                message: "Missing required register fields: id, name, companyName, document, state, city",
            });
        }
        // Criar job e retornar imediatamente (202 Accepted)
        const jobId = jobQueue_service_1.jobQueueService.enqueue("create-bemchat", {
            name: body.name,
            type: body.type,
            register: body.register,
            port: body.port,
        });
        // Iniciar processamento em background usando setImmediate
        // Isso garante que o job execute em outro ciclo do event loop
        setImmediate(() => {
            this.startCreateJob(jobId, body.name, body.type, body.register, body.port).catch((error) => {
                console.error(`[BemChatController] Unexpected error in startCreateJob for ${jobId}:`, error);
                jobQueue_service_1.jobQueueService.failJob(jobId, error.message || "Unexpected error");
            });
        });
        // Retornar 202 Accepted com jobId
        c.status(202);
        return c.json({
            jobId,
            message: "BemChat creation started",
            statusUrl: `/status/${jobId}`,
        });
    };
    /**
     * Processa a criação do BemChat em background
     */
    async startCreateJob(jobId, name, type, register, port) {
        console.log(`[BemChatController] Starting job ${jobId} for ${name} (${type})`);
        try {
            // Atualizar status para "running"
            jobQueue_service_1.jobQueueService.updateJob(jobId, { status: "running", progress: 10 });
            console.log(`[BemChatController] Job ${jobId} - Status: running (10%)`);
            // Simular progresso durante a criação (a cada 30 segundos)
            const progressInterval = setInterval(() => {
                const currentJob = jobQueue_service_1.jobQueueService.getJob(jobId);
                if (currentJob &&
                    currentJob.status === "running" &&
                    currentJob.progress < 90) {
                    const newProgress = Math.min(currentJob.progress + 10, 90);
                    jobQueue_service_1.jobQueueService.updateJob(jobId, { progress: newProgress });
                    console.log(`[BemChatController] Job ${jobId} - Progress: ${newProgress}%`);
                }
            }, 30000); // A cada 30 segundos
            // Executar criação (operação longa - 5+ minutos)
            console.log(`[BemChatController] Job ${jobId} - Calling createBemChat...`);
            const result = await this.bemchatService.createBemChat(name, type, register, port);
            console.log(`[BemChatController] Job ${jobId} - createBemChat returned:`, result);
            // Parar o progresso simulado
            clearInterval(progressInterval);
            // Verificar resultado e atualizar job
            if (result.startsWith("Error") || result.includes("not found")) {
                console.error(`[BemChatController] Job ${jobId} - Failed:`, result);
                jobQueue_service_1.jobQueueService.failJob(jobId, result);
            }
            else {
                console.log(`[BemChatController] Job ${jobId} - Completed successfully`);
                jobQueue_service_1.jobQueueService.completeJob(jobId, result);
            }
        }
        catch (error) {
            const errorMessage = error instanceof Error ? error.message : "Unknown error";
            console.error(`[BemChatController] Job ${jobId} - Exception:`, error);
            jobQueue_service_1.jobQueueService.failJob(jobId, errorMessage);
        }
    }
    /**
     * Retorna o status de um job
     */
    getStatus = async (c) => {
        const { jobId } = c.req.param();
        if (!jobId) {
            c.status(400);
            return c.json({ message: "jobId is required" });
        }
        const job = jobQueue_service_1.jobQueueService.getJob(jobId);
        if (!job) {
            c.status(404);
            return c.json({ message: "Job not found" });
        }
        return c.json({
            jobId: job.id,
            status: job.status,
            progress: job.progress,
            result: job.result,
            error: job.error,
            createdAt: job.createdAt.toISOString(),
            completedAt: job.completedAt?.toISOString(),
        });
    };
    deleteOne = async (c) => {
        const { name } = c.req.param();
        if (!name) {
            c.status(400);
            return c.json({ message: "Instance name is required" });
        }
        const result = await this.bemchatService.deleteBemChat(name);
        if (result.startsWith("Error") || result.includes("not found")) {
            c.status(500);
            return c.json({ message: result });
        }
        return c.json({ result });
    };
    updateOne = async (c) => {
        const { name } = await (0, parseRequestBody_1.parseRequestBody)(c);
        if (!name) {
            c.status(400);
            return c.json({ message: "request body is empty" });
        }
        const result = await this.bemchatService.updateBemChat(name);
        return c.json({ result });
    };
    updateAll = async (c) => {
        const { bemChatType } = c.req.query();
        const result = await this.bemchatService.updateBemChats(bemChatType);
        return c.json({ result });
    };
    updateDatabase = async (c) => {
        const { name } = c.req.param();
        if (!name) {
            c.status(400);
            return c.json({ message: "Instance name is required" });
        }
        const result = await this.bemchatService.updateDatabase(name);
        return c.json({ result });
    };
    list = async (c) => {
        const result = this.bemchatService.listBemChats();
        return c.json({ result });
    };
    restartInstance = async (c) => {
        const body = (await (0, parseRequestBody_1.parseRequestBody)(c));
        // Validate required fields
        if (!body.name || !body.type) {
            c.status(400);
            return c.json({
                message: "Missing required fields: name and type",
            });
        }
        // Validate type
        if (body.type !== "api" && body.type !== "platform") {
            c.status(400);
            return c.json({
                message: "Invalid type: must be 'api' or 'platform'",
            });
        }
        // Validate name format (no underscores)
        if (body.name.includes("_")) {
            c.status(400);
            return c.json({
                message: "Invalid instance name: underscore (_) is not allowed. Use hyphen (-) instead.",
            });
        }
        // Validate name pattern (lowercase, numbers, hyphens only)
        const namePattern = /^[a-z0-9-]+$/;
        if (!namePattern.test(body.name)) {
            c.status(400);
            return c.json({
                message: "Invalid instance name: only lowercase letters, numbers, and hyphens are allowed",
            });
        }
        // Create job
        const jobId = jobQueue_service_1.jobQueueService.enqueue("restart-bemchat", {
            name: body.name,
            type: body.type,
        });
        // Process in background
        setImmediate(() => {
            this.startRestartJob(jobId, body.name, body.type).catch((error) => {
                console.error(`[BemChatController] Unexpected error in startRestartJob for ${jobId}:`, error);
                jobQueue_service_1.jobQueueService.failJob(jobId, error.message || "Unexpected error");
            });
        });
        // Return 202 Accepted
        c.status(202);
        return c.json({
            jobId,
            message: "Restart job created successfully",
            statusUrl: `/status/${jobId}`,
        });
    };
    async startRestartJob(jobId, name, type) {
        console.log(`[BemChatController] Starting restart job ${jobId} for ${name} (${type})`);
        try {
            // Update to running
            jobQueue_service_1.jobQueueService.updateJob(jobId, { status: "running", progress: 10 });
            console.log(`[BemChatController] Job ${jobId} - Status: running (10%)`);
            // Execute restart (fast operation)
            const result = await this.bemchatService.restartInstance(name, type);
            console.log(`[BemChatController] Job ${jobId} - restartInstance returned:`, result);
            // Update progress before completion
            jobQueue_service_1.jobQueueService.updateJob(jobId, { progress: 90 });
            // Check result
            if (result.startsWith("Error") || result.includes("not found")) {
                console.error(`[BemChatController] Job ${jobId} - Failed:`, result);
                jobQueue_service_1.jobQueueService.failJob(jobId, result);
            }
            else {
                console.log(`[BemChatController] Job ${jobId} - Completed successfully`);
                jobQueue_service_1.jobQueueService.completeJob(jobId, result);
            }
        }
        catch (error) {
            const errorMessage = error instanceof Error ? error.message : "Unknown error";
            console.error(`[BemChatController] Job ${jobId} - Exception:`, error);
            jobQueue_service_1.jobQueueService.failJob(jobId, errorMessage);
        }
    }
}
exports.BemChatController = BemChatController;
const bemchatController = new BemChatController(bemchat_service_1.bemChatService);
exports.bemchatController = bemchatController;
//# sourceMappingURL=bemchat.controller.js.map