"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createDirectory = createDirectory;
exports.writeFile = writeFile;
exports.deleteDirectory = deleteDirectory;
exports.fileExists = fileExists;
exports.readFile = readFile;
const node_fs_1 = require("node:fs");
const promises_1 = __importDefault(require("node:fs/promises"));
const node_path_1 = __importDefault(require("node:path"));
/**
 * Cria um diretório de forma assíncrona
 * @param dirPath - Caminho do diretório a ser criado
 * @param recursive - Se true, cria diretórios pais se necessário
 */
async function createDirectory(dirPath, recursive = true) {
    try {
        await promises_1.default.mkdir(dirPath, { recursive });
    }
    catch (error) {
        if (error instanceof Error) {
            throw new Error(`Failed to create directory ${dirPath}: ${error.message}`);
        }
        throw new Error(`Failed to create directory ${dirPath}: Unknown error`);
    }
}
/**
 * Escreve conteúdo em um arquivo de forma assíncrona
 * @param filePath - Caminho do arquivo
 * @param content - Conteúdo a ser escrito
 */
async function writeFile(filePath, content) {
    try {
        // Garantir que o diretório pai existe
        const dir = node_path_1.default.dirname(filePath);
        if (!(0, node_fs_1.existsSync)(dir)) {
            await createDirectory(dir, true);
        }
        await promises_1.default.writeFile(filePath, content, "utf-8");
    }
    catch (error) {
        if (error instanceof Error) {
            throw new Error(`Failed to write file ${filePath}: ${error.message}`);
        }
        throw new Error(`Failed to write file ${filePath}: Unknown error`);
    }
}
/**
 * Remove um diretório de forma assíncrona
 * @param dirPath - Caminho do diretório a ser removido
 * @param recursive - Se true, remove recursivamente todo o conteúdo
 */
async function deleteDirectory(dirPath, recursive = true) {
    try {
        await promises_1.default.rm(dirPath, { recursive, force: true });
    }
    catch (error) {
        if (error instanceof Error) {
            throw new Error(`Failed to delete directory ${dirPath}: ${error.message}`);
        }
        throw new Error(`Failed to delete directory ${dirPath}: Unknown error`);
    }
}
/**
 * Verifica se um arquivo ou diretório existe
 * @param filePath - Caminho do arquivo ou diretório
 * @returns true se existe, false caso contrário
 */
async function fileExists(filePath) {
    try {
        await promises_1.default.access(filePath);
        return true;
    }
    catch {
        return false;
    }
}
/**
 * Lê o conteúdo de um arquivo de forma assíncrona
 * @param filePath - Caminho do arquivo
 * @returns Conteúdo do arquivo como string
 */
async function readFile(filePath) {
    try {
        return await promises_1.default.readFile(filePath, "utf-8");
    }
    catch (error) {
        if (error instanceof Error) {
            throw new Error(`Failed to read file ${filePath}: ${error.message}`);
        }
        throw new Error(`Failed to read file ${filePath}: Unknown error`);
    }
}
//# sourceMappingURL=fileSystem.js.map