"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.jobQueueService = void 0;
const node_crypto_1 = require("node:crypto");
class JobQueueService {
    jobs = new Map();
    cleanupInterval = null;
    constructor() {
        // Iniciar limpeza automática de jobs antigos a cada 1 hora
        this.startCleanup();
    }
    /**
     * Cria um novo job e adiciona à fila
     */
    enqueue(type, data) {
        const jobId = (0, node_crypto_1.randomUUID)();
        const job = {
            id: jobId,
            type,
            status: "pending",
            progress: 0,
            data,
            createdAt: new Date(),
        };
        this.jobs.set(jobId, job);
        console.log(`[JobQueue] Job ${jobId} enqueued (${type})`);
        return jobId;
    }
    /**
     * Obtém um job pelo ID
     */
    getJob(jobId) {
        return this.jobs.get(jobId);
    }
    /**
     * Atualiza parcialmente um job
     */
    updateJob(jobId, updates) {
        const job = this.jobs.get(jobId);
        if (job) {
            Object.assign(job, updates);
            console.log(`[JobQueue] Job ${jobId} updated - Status: ${job.status}, Progress: ${job.progress}%`);
        }
    }
    /**
     * Marca um job como completado com sucesso
     */
    completeJob(jobId, result) {
        this.updateJob(jobId, {
            status: "completed",
            result,
            progress: 100,
            completedAt: new Date(),
        });
        console.log(`[JobQueue] Job ${jobId} completed successfully`);
    }
    /**
     * Marca um job como falho
     */
    failJob(jobId, error) {
        this.updateJob(jobId, {
            status: "failed",
            error,
            completedAt: new Date(),
        });
        console.error(`[JobQueue] Job ${jobId} failed: ${error}`);
    }
    /**
     * Remove jobs antigos (mais de 24 horas)
     */
    cleanupOldJobs(maxAgeMs = 24 * 60 * 60 * 1000) {
        const now = Date.now();
        let removedCount = 0;
        for (const [id, job] of this.jobs.entries()) {
            const jobAge = now - job.createdAt.getTime();
            if (jobAge > maxAgeMs) {
                this.jobs.delete(id);
                removedCount++;
            }
        }
        if (removedCount > 0) {
            console.log(`[JobQueue] Cleaned up ${removedCount} old jobs`);
        }
        return removedCount;
    }
    /**
     * Inicia limpeza automática periódica
     */
    startCleanup() {
        // Executar limpeza a cada 1 hora
        this.cleanupInterval = setInterval(() => {
            this.cleanupOldJobs();
        }, 60 * 60 * 1000); // 1 hora
    }
    /**
     * Para a limpeza automática
     */
    stopCleanup() {
        if (this.cleanupInterval) {
            clearInterval(this.cleanupInterval);
            this.cleanupInterval = null;
        }
    }
    /**
     * Retorna estatísticas dos jobs
     */
    getStats() {
        const jobs = Array.from(this.jobs.values());
        return {
            total: jobs.length,
            pending: jobs.filter((j) => j.status === "pending").length,
            running: jobs.filter((j) => j.status === "running").length,
            completed: jobs.filter((j) => j.status === "completed").length,
            failed: jobs.filter((j) => j.status === "failed").length,
        };
    }
    /**
     * Lista todos os jobs (para debug)
     */
    listJobs() {
        return Array.from(this.jobs.values());
    }
}
// Exportar instância única (Singleton)
exports.jobQueueService = new JobQueueService();
//# sourceMappingURL=jobQueue.service.js.map