"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BemChatPlataform = void 0;
const cmdFunctions_1 = require("../../../utils/cmdFunctions");
const configGenerator_1 = require("../../../utils/configGenerator");
const node_fs_1 = __importDefault(require("node:fs"));
const node_path_1 = __importDefault(require("node:path"));
class BemChatPlataform {
    name;
    port;
    path;
    constructor(props) {
        this.name = props.name;
        this.port = props.port;
        this.path = props.path || "";
    }
    async getAvailableProcesses() {
        try {
            const listAllCommand = `pm2 jlist | jq -r '.[] | .name'`;
            const result = await (0, cmdFunctions_1.execCommand)(listAllCommand, true);
            if (result && result.trim()) {
                return result
                    .trim()
                    .split("\n")
                    .filter((name) => name.length > 0);
            }
            return [];
        }
        catch (error) {
            console.error(`Error listing PM2 processes:`, error);
            return [];
        }
    }
    async findProcessByPattern(pattern) {
        try {
            const findCommand = `pm2 jlist | jq -r '.[] | select(.name | startswith("${pattern}")) | .name' | head -n 1`;
            const result = await (0, cmdFunctions_1.execCommand)(findCommand, true);
            return result && result.trim() ? result.trim() : null;
        }
        catch (error) {
            console.error(`Error finding process with pattern "${pattern}":`, error);
            return null;
        }
    }
    async createBemChat(register, customPort) {
        // Scripts ficam no diretório home (~) do servidor
        const homeDir = process.env.HOME || "~";
        const createScriptPath = node_path_1.default.join(homeDir, "create-bemchat.sh");
        // Verificar se o script de criação existe
        if (!node_fs_1.default.existsSync(createScriptPath)) {
            return `Error: Script create-bemchat.sh not found at ${createScriptPath}`;
        }
        // Verificar se a instância já existe
        if (node_fs_1.default.existsSync(this.path)) {
            return `Error: Instance ${this.name} already exists at ${this.path}`;
        }
        try {
            // Preparar comando com porta customizada se fornecida
            let createCommand = `bash ./create-bemchat.sh ${this.name}`;
            if (customPort) {
                // Passar porta customizada como variável de ambiente
                createCommand = `CUSTOM_PORT=${customPort} ${createCommand}`;
            }
            // Executar script de criação com múltiplas interrupções
            const interruptions = [
                {
                    message: "criação da aplicação",
                    answer: "S I M",
                },
                {
                    message: "Select the appropriate number",
                    answer: "1", // Reinstalar certificado existente
                },
            ];
            // Criar ambiente limpo sem PORT herdado do Guardian
            const cleanEnv = { ...process.env };
            delete cleanEnv.PORT;
            delete cleanEnv.CUSTOM_PORT;
            const result = await (0, cmdFunctions_1.spawnCmd)(createCommand, interruptions, {
                cwd: homeDir,
                env: cleanEnv,
            });
            if (typeof result === "string") {
                const returnCode = result.split("_")[1];
                if (returnCode !== "0") {
                    return `Error creating BemChat ${this.name}: Script exited with code ${returnCode}`;
                }
            }
            else {
                return `Error creating BemChat ${this.name}: Unexpected script response`;
            }
            // Verificar se a pasta foi criada
            if (!node_fs_1.default.existsSync(this.path)) {
                return `Error: Instance folder was not created at ${this.path}`;
            }
            // Gerar arquivo config.json
            await (0, configGenerator_1.generateConfigJson)(register, this.path, this.name, "plataform");
            // Aguardar um momento para garantir que o arquivo foi escrito no disco
            await new Promise((resolve) => setTimeout(resolve, 3000));
            // Reiniciar backend para carregar o novo config.json
            try {
                console.log(`Restarting backend process for ${this.name}...`);
                const backendPattern = `${this.name}_back_`;
                const processName = await this.findProcessByPattern(backendPattern);
                if (processName) {
                    const restartCommand = `pm2 restart "${processName}"`;
                    await (0, cmdFunctions_1.execCommand)(restartCommand, true);
                    console.log(`Backend process ${processName} restarted successfully`);
                }
                else {
                    // Listar processos disponíveis para debug
                    const availableProcesses = await this.getAvailableProcesses();
                    console.warn(`Backend process for ${this.name} not found in PM2`);
                    console.warn(`Pattern searched: ${backendPattern}`);
                    console.warn(`Available PM2 processes:`, availableProcesses);
                }
            }
            catch (error) {
                // Não falhar a criação se o restart falhar
                console.warn(`Warning: Could not restart backend process for ${this.name}:`, error);
                const availableProcesses = await this.getAvailableProcesses();
                console.warn(`Available PM2 processes:`, availableProcesses);
                console.warn(`You may need to restart it manually using: pm2 restart ${this.name}_back_*`);
            }
            // Aguardar um momento para garantir que o arquivo foi escrito no disco
            await new Promise((resolve) => setTimeout(resolve, 5000));
            // Reiniciar app e recarregar processo de dados
            try {
                console.log(`Restarting app and reloading data process for ${this.name}...`);
                const backendPattern = `${this.name}_back_`;
                const frontendPattern = `${this.name}_front_`;
                const processBackendName = await this.findProcessByPattern(backendPattern);
                const processFrontendName = await this.findProcessByPattern(frontendPattern);
                if (processBackendName) {
                    const restartCommand = `pm2 restart "${processBackendName}"`;
                    await (0, cmdFunctions_1.execCommand)(restartCommand, true);
                    console.log(`Backend process ${processBackendName} restarted successfully`);
                }
                else {
                    const availableProcesses = await this.getAvailableProcesses();
                    console.warn(`Backend process for ${this.name} not found in PM2`);
                    console.warn(`Pattern searched: ${backendPattern}`);
                    console.warn(`Available PM2 processes:`, availableProcesses);
                }
                if (processFrontendName) {
                    const restartCommand = `pm2 restart "${processFrontendName}"`;
                    await (0, cmdFunctions_1.execCommand)(restartCommand, true);
                    console.log(`Frontend process ${processFrontendName} restarted successfully`);
                }
                else {
                    const availableProcesses = await this.getAvailableProcesses();
                    console.warn(`Frontend process for ${this.name} not found in PM2`);
                    console.warn(`Pattern searched: ${frontendPattern}`);
                    console.warn(`Available PM2 processes:`, availableProcesses);
                }
            }
            catch (error) {
                // Não falhar a criação se o restart falhar
                console.warn(`Warning: Could not restart processes for ${this.name}:`, error);
                const availableProcesses = await this.getAvailableProcesses();
                console.warn(`Available PM2 processes:`, availableProcesses);
                console.warn(`You may need to restart them manually using: pm2 restart ${this.name}_back_* ${this.name}_front_*`);
            }
            return `BemChat ${this.name} created successfully`;
        }
        catch (error) {
            if (error instanceof Error) {
                return `Error creating BemChat ${this.name}: ${error.message}`;
            }
            return `Error creating BemChat ${this.name}: Unknown error`;
        }
    }
    getBemChatInfo() {
        return { name: this.name, port: this.port, path: this.path };
    }
    async updateBemChat() {
        if (!node_fs_1.default.existsSync(this.path)) {
            return `folder ${this.path} not found`;
        }
        const updateScriptPath = node_path_1.default.resolve(this.path, "../", "update-bemchat.sh");
        if (!node_fs_1.default.existsSync(updateScriptPath)) {
            return "file update-bemchat.sh not found";
        }
        const confirmInterruption = {
            message: "Para prosseguir para a atualização digite",
            answer: "S I M",
        };
        const result = await (0, cmdFunctions_1.spawnCmd)(`./update-bemchat.sh ${this.name}`, confirmInterruption, { cwd: node_path_1.default.dirname(updateScriptPath) });
        if (typeof result === "string") {
            const returnCodeFrontend = result.split("_")[1];
            if (returnCodeFrontend !== "0") {
                return `Error updating BemChat ${this.name}`;
            }
            return `BemChat ${this.name} updated`;
        }
        return "Error while updating bemchat";
    }
    async updateDatabase(sourceScriptPath) {
        const targetScriptPath = node_path_1.default.join(this.path, "backend", "database", "update-db.sh");
        if (!node_fs_1.default.existsSync(targetScriptPath)) {
            if (!node_fs_1.default.existsSync(sourceScriptPath)) {
                return `Source script not found at ${sourceScriptPath}`;
            }
            try {
                node_fs_1.default.mkdirSync(node_path_1.default.dirname(targetScriptPath), { recursive: true });
                node_fs_1.default.copyFileSync(sourceScriptPath, targetScriptPath);
                node_fs_1.default.chmodSync(targetScriptPath, 0o755);
            }
            catch (error) {
                if (error instanceof Error)
                    return `Failed to copy script: ${error.message}`;
                return "Failed to copy script due to an unknown error.";
            }
        }
        const result = await (0, cmdFunctions_1.spawnCmd)("./update-db.sh", undefined, {
            cwd: node_path_1.default.dirname(targetScriptPath),
        });
        if (typeof result === "string") {
            const returnCode = result.split("_")[1];
            if (returnCode !== "0") {
                return `Error updating database for ${this.name}`;
            }
            return `Database for ${this.name} updated successfully`;
        }
        return `An unknown error occurred while updating database for ${this.name}`;
    }
    async deleteBemChat() {
        // Scripts ficam no diretório home (~) do servidor
        const homeDir = process.env.HOME || "~";
        const deleteScriptPath = node_path_1.default.join(homeDir, "delete-bemchat.sh");
        // Verificar se o script de remoção existe
        if (!node_fs_1.default.existsSync(deleteScriptPath)) {
            return `Error: Script delete-bemchat.sh not found at ${deleteScriptPath}`;
        }
        // Verificar se a instância existe
        if (!node_fs_1.default.existsSync(this.path)) {
            return `Error: Instance ${this.name} does not exist at ${this.path}`;
        }
        try {
            // Executar script de remoção
            const confirmInterruption = {
                message: "confirmar a remoção",
                answer: "C O N F I R M O",
            };
            const result = await (0, cmdFunctions_1.spawnCmd)(`bash ./delete-bemchat.sh ${this.name}`, confirmInterruption, { cwd: homeDir });
            if (typeof result === "string") {
                const returnCode = result.split("_")[1];
                if (returnCode !== "0") {
                    return `Error deleting BemChat ${this.name}: Script exited with code ${returnCode}`;
                }
            }
            else {
                return `Error deleting BemChat ${this.name}: Unexpected script response`;
            }
            // Verificar se a pasta foi removida
            if (node_fs_1.default.existsSync(this.path)) {
                return `Error: Instance folder was not removed at ${this.path}`;
            }
            return `BemChat ${this.name} deleted successfully`;
        }
        catch (error) {
            if (error instanceof Error) {
                return `Error deleting BemChat ${this.name}: ${error.message}`;
            }
            return `Error deleting BemChat ${this.name}: Unknown error`;
        }
    }
    async restart() {
        // Verify instance exists
        if (!node_fs_1.default.existsSync(this.path)) {
            return `Error: Platform instance ${this.name} not found at ${this.path}`;
        }
        try {
            const results = [];
            // Find and restart backend
            const backendProcess = await this.findProcessByPattern(`${this.name}_back_`);
            if (backendProcess) {
                await (0, cmdFunctions_1.execCommand)(`pm2 restart "${backendProcess}"`, true);
                results.push(`Backend ${backendProcess} restarted`);
            }
            else {
                results.push(`Warning: Backend process not found`);
            }
            // Find and restart frontend
            const frontendProcess = await this.findProcessByPattern(`${this.name}_front_`);
            if (frontendProcess) {
                await (0, cmdFunctions_1.execCommand)(`pm2 restart "${frontendProcess}"`, true);
                results.push(`Frontend ${frontendProcess} restarted`);
            }
            else {
                results.push(`Warning: Frontend process not found`);
            }
            // Fail if no processes found
            if (!backendProcess && !frontendProcess) {
                return `Error: No PM2 processes found for Platform ${this.name}`;
            }
            return `Platform ${this.name} restarted successfully. ${results.join(", ")}`;
        }
        catch (error) {
            if (error instanceof Error) {
                return `Error restarting Platform ${this.name}: ${error.message}`;
            }
            return `Error restarting Platform ${this.name}: Unknown error`;
        }
    }
}
exports.BemChatPlataform = BemChatPlataform;
