"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.pullImage = pullImage;
exports.composeUp = composeUp;
exports.composeDown = composeDown;
exports.composeRestart = composeRestart;
const node_path_1 = __importDefault(require("node:path"));
const cmdFunctions_1 = require("./cmdFunctions");
/**
 * Faz pull de uma imagem Docker com autenticação inline
 * Usa DOCKER_CONFIG para salvar credenciais em /opt/guardian/.docker-config
 * @param imageName - Nome completo da imagem (registry/image:tag)
 */
async function pullImage(imageName) {
    try {
        const username = process.env.DOCKER_USERNAME;
        const password = process.env.DOCKER_PASSWORD;
        const dockerConfigDir = "/opt/guardian/.docker-config";
        // Extrair registry do nome da imagem
        const registry = imageName.split("/")[0];
        // Se for registry privado e temos credenciais, fazer pull autenticado
        if (registry.includes(".") && username && password) {
            console.log(`[dockerManager] Pulling image with authentication: ${imageName}`);
            // Usar DOCKER_CONFIG para diretório próprio (evita problemas de permissão)
            const command = `DOCKER_CONFIG=${dockerConfigDir} sh -c 'echo "${password}" | docker login ${registry} -u ${username} --password-stdin && docker pull ${imageName} && docker logout ${registry}'`;
            await (0, cmdFunctions_1.execCommand)(command, false);
            console.log(`[dockerManager] Successfully pulled image: ${imageName}`);
        }
        else {
            // Pull sem autenticação (imagem pública ou sem credenciais)
            console.log(`[dockerManager] Pulling image: ${imageName}`);
            await (0, cmdFunctions_1.execCommand)(`docker pull ${imageName}`, false);
            console.log(`[dockerManager] Successfully pulled image: ${imageName}`);
        }
    }
    catch (error) {
        console.error(`[dockerManager] Error pulling image:`, error);
        throw new Error(`Failed to pull Docker image ${imageName}: ${error instanceof Error ? error.message : "Unknown error"}`);
    }
}
/**
 * Inicia containers usando docker compose
 * @param composePath - Caminho para o arquivo docker-compose.yaml
 * @param projectName - Nome do projeto
 */
async function composeUp(composePath, projectName) {
    try {
        const composeDir = node_path_1.default.dirname(composePath);
        const dockerConfigDir = "/opt/guardian/.docker-config";
        console.log(`[dockerManager] Starting containers for project: ${projectName}`);
        // Usar DOCKER_CONFIG para evitar problemas de permissão
        await (0, cmdFunctions_1.execCommand)(`cd ${composeDir} && DOCKER_CONFIG=${dockerConfigDir} docker compose --project-name ${projectName} up -d`, false);
        console.log(`[dockerManager] Successfully started containers for project: ${projectName}`);
    }
    catch (error) {
        console.error(`[dockerManager] Error starting containers:`, error);
        throw new Error(`Failed to start Docker containers for ${projectName}: ${error instanceof Error ? error.message : "Unknown error"}`);
    }
}
/**
 * Para e remove containers usando docker compose
 * @param composePath - Caminho para o arquivo docker-compose.yaml
 * @param projectName - Nome do projeto
 */
async function composeDown(composePath, projectName) {
    try {
        const composeDir = node_path_1.default.dirname(composePath);
        const dockerConfigDir = "/opt/guardian/.docker-config";
        console.log(`[dockerManager] Stopping containers for project: ${projectName}`);
        await (0, cmdFunctions_1.execCommand)(`cd ${composeDir} && DOCKER_CONFIG=${dockerConfigDir} docker compose --project-name ${projectName} down`, false);
        console.log(`[dockerManager] Successfully stopped containers for project: ${projectName}`);
    }
    catch (error) {
        console.error(`[dockerManager] Error stopping containers:`, error);
        throw new Error(`Failed to stop Docker containers for ${projectName}: ${error instanceof Error ? error.message : "Unknown error"}`);
    }
}
/**
 * Reinicia containers usando docker compose
 * @param composePath - Caminho para o arquivo docker-compose.yaml
 * @param projectName - Nome do projeto
 */
async function composeRestart(composePath, projectName) {
    try {
        const composeDir = node_path_1.default.dirname(composePath);
        const dockerConfigDir = "/opt/guardian/.docker-config";
        console.log(`[dockerManager] Restarting containers for project: ${projectName}`);
        await (0, cmdFunctions_1.execCommand)(`cd ${composeDir} && DOCKER_CONFIG=${dockerConfigDir} docker compose --project-name ${projectName} restart`, false);
        console.log(`[dockerManager] Successfully restarted containers for project: ${projectName}`);
    }
    catch (error) {
        console.error(`[dockerManager] Error restarting containers:`, error);
        throw new Error(`Failed to restart Docker containers for ${projectName}: ${error instanceof Error ? error.message : "Unknown error"}`);
    }
}
