"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const dotenv_1 = __importDefault(require("dotenv"));
const node_fs_1 = __importDefault(require("node:fs"));
const node_path_1 = __importDefault(require("node:path"));
const instanceDatabase_1 = require("./instanceDatabase");
// Load environment variables
dotenv_1.default.config();
const ignoreFolders = ["BACKUPS", "_latest"];
async function syncDatabase() {
    console.log("🚀 Starting database synchronization...");
    const apiPath = process.env.API_INSTANCES_PATH || "/srv/bemchat/api";
    const plataformPath = process.env.PLATAFORM_INSTANCES_PATH || "/srv/bemchat/bemchat";
    console.log(`📂 API Path: ${apiPath}`);
    console.log(`📂 Platform Path: ${plataformPath}`);
    const db = (0, instanceDatabase_1.getDatabase)();
    // ==========================
    // SYNC API INSTANCES
    // ==========================
    if (node_fs_1.default.existsSync(apiPath)) {
        const apiDirs = node_fs_1.default
            .readdirSync(apiPath, { withFileTypes: true })
            .filter((dirent) => dirent.isDirectory() && !ignoreFolders.includes(dirent.name))
            .map((dirent) => dirent.name);
        console.log(`\nFound ${apiDirs.length} API directories.`);
        for (const dirName of apiDirs) {
            const instancePath = node_path_1.default.join(apiPath, dirName);
            const composePath = node_path_1.default.join(instancePath, "docker-compose.yaml");
            try {
                if (node_fs_1.default.existsSync(composePath)) {
                    const composeContent = node_fs_1.default.readFileSync(composePath, "utf-8");
                    // Extract port: - "3001:3000"
                    const portMatch = composeContent.match(/- "(\d+):3000"/);
                    if (portMatch && portMatch[1]) {
                        const port = Number.parseInt(portMatch[1], 10);
                        console.log(`[API] Syncing ${dirName} (Port: ${port})...`);
                        db.addInstance({
                            name: dirName,
                            port: port,
                            type: "api",
                            path: instancePath,
                        });
                    }
                    else {
                        console.warn(`⚠️  [API] Could not find port in docker-compose for ${dirName}`);
                    }
                }
                else {
                    console.warn(`⚠️  [API] docker-compose.yaml not found for ${dirName}`);
                }
            }
            catch (error) {
                console.error(`❌ [API] Error processing ${dirName}:`, error);
            }
        }
    }
    else {
        console.warn(`⚠️  API directory not found at ${apiPath}`);
    }
    // ==========================
    // SYNC PLATFORM INSTANCES
    // ==========================
    if (node_fs_1.default.existsSync(plataformPath)) {
        const platformDirs = node_fs_1.default
            .readdirSync(plataformPath, { withFileTypes: true })
            .filter((dirent) => dirent.isDirectory() && !ignoreFolders.includes(dirent.name))
            .map((dirent) => dirent.name);
        console.log(`\nFound ${platformDirs.length} Platform directories.`);
        for (const dirName of platformDirs) {
            const instancePath = node_path_1.default.join(plataformPath, dirName);
            // Attempt to find port in backend/.env
            const backendEnvPath = node_path_1.default.join(instancePath, "backend", ".env");
            let port = 0;
            try {
                if (node_fs_1.default.existsSync(backendEnvPath)) {
                    const envContent = node_fs_1.default.readFileSync(backendEnvPath, "utf-8");
                    const portMatch = envContent.match(/^PORT=(\d+)/m);
                    if (portMatch && portMatch[1]) {
                        port = Number.parseInt(portMatch[1], 10);
                    }
                }
                if (port === 0) {
                    // Try looking for 'app/.env' or other common locations if needed?
                    // checking 'app/backend/.env' just in case structure varies
                    const altBackendEnv = node_path_1.default.join(instancePath, "app", "backend", ".env");
                    if (node_fs_1.default.existsSync(altBackendEnv)) {
                        const envContent = node_fs_1.default.readFileSync(altBackendEnv, "utf-8");
                        const portMatch = envContent.match(/^PORT=(\d+)/m);
                        if (portMatch && portMatch[1]) {
                            port = Number.parseInt(portMatch[1], 10);
                        }
                    }
                }
                if (port > 0) {
                    console.log(`[Platform] Syncing ${dirName} (Port: ${port})...`);
                    db.addInstance({
                        name: dirName,
                        port: port,
                        type: "plataform",
                        path: instancePath,
                    });
                }
                else {
                    // If we can't find the port, we might still want to add it but we can't guarantee port uniqueness.
                    // However, the DB requires a UNIQUE port. Inserting 0 might fail if multiple are 0.
                    // Also, isPortInUse uses this table. If we miss the port, we miss the protection.
                    // But if we insert a dummy port, we might block a valid one.
                    console.warn(`⚠️  [Platform] Could not determine port for ${dirName}. Skipping DB sync for safety.`);
                }
            }
            catch (error) {
                console.error(`❌ [Platform] Error processing ${dirName}:`, error);
            }
        }
    }
    else {
        console.warn(`⚠️  Platform directory not found at ${plataformPath}`);
    }
    console.log("\n✅ Database synchronization complete.");
    db.close();
}
syncDatabase().catch((err) => {
    console.error("Fatal Error:", err);
    process.exit(1);
});
