"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BemChatService = exports.bemChatService = void 0;
const node_fs_1 = __importDefault(require("node:fs"));
const node_path_1 = __importDefault(require("node:path"));
const bemchat_api_1 = require("./classes/bemchat-api");
const bemchat_plataform_1 = require("./classes/bemchat-plataform");
const ignoreFolders = ["BACKUPS", "_latest"];
class BemChatService {
    bemchats = [];
    plataformPath;
    apiPath;
    constructor() {
        this.plataformPath = process.env.PLATAFORM_INSTANCES_PATH || "/srv/bemchat/platform";
        this.apiPath = process.env.API_INSTANCES_PATH || "/srv/bemchat/api";
        this.loadBemChats();
    }
    loadBemChats() {
        this.bemchats = []; // Limpar lista antes de recarregar
        // Arrays para rastrear instancias encontradas no sistema de arquivos
        const foundInstances = [];
        // Carregar plataformas
        const plataformInstances = this.loadFromDirectory(this.plataformPath, "plataform");
        if (plataformInstances)
            foundInstances.push(...plataformInstances);
        // Carregar APIs
        const apiInstances = this.loadFromDirectory(this.apiPath, "api");
        if (apiInstances)
            foundInstances.push(...apiInstances);
        // Sincronizar REMOÇÃO com banco de dados (apenas para apis que estao no banco)
        // Instancias de plataforma nao sao salvas no banco atualmente pelo codigo original, 
        // mas se fossem, a logica seria a mesma. O foco aqui é limpar lixo do banco.
        try {
            const { getDatabase } = require("../../utils/instanceDatabase");
            const db = getDatabase();
            const allDbInstances = db.getAllInstances();
            for (const dbInst of allDbInstances) {
                if (!foundInstances.includes(dbInst.name)) {
                    console.log(`[BemChatService] Removing stale instance from database: ${dbInst.name}`);
                    db.removeInstance(dbInst.name);
                }
            }
        }
        catch (error) {
            console.warn("[BemChatService] Could not sync removal with database:", error);
        }
    }
    loadFromDirectory(directory, instanceType) {
        const foundNames = [];
        const pathExist = node_fs_1.default.existsSync(directory);
        if (!pathExist) {
            console.log(`[BemChatService] ${instanceType} directory not found: ${directory} - skipping`);
            return [];
        }
        const bemchatFiles = node_fs_1.default
            .readdirSync(directory, { withFileTypes: true })
            .filter((file) => file.isDirectory() && !ignoreFolders.includes(file.name))
            .map((dir) => dir.name);
        console.log(`[BemChatService] Loading ${instanceType} instances from ${directory}:`, bemchatFiles);
        for (const file of bemchatFiles) {
            const instancePath = node_path_1.default.join(directory, file);
            foundNames.push(file);
            if (instanceType === "api") {
                // Para API: o nome da pasta é o nome da instância
                // Precisamos ler a porta do docker-compose.yaml
                let port = 0;
                try {
                    const composePath = node_path_1.default.join(instancePath, "docker-compose.yaml");
                    if (node_fs_1.default.existsSync(composePath)) {
                        const composeContent = node_fs_1.default.readFileSync(composePath, "utf-8");
                        // Extrair porta do formato: - "3001:3000"
                        const portMatch = composeContent.match(/- "(\d+):3000"/);
                        if (portMatch) {
                            port = Number.parseInt(portMatch[1], 10);
                        }
                    }
                }
                catch (error) {
                    console.warn(`[BemChatService] Could not read port for ${file}:`, error);
                }
                const bemchat = new bemchat_api_1.BemChatApi({
                    name: file, // Nome da pasta é o nome da instância
                    port: port,
                    path: instancePath,
                });
                const info = bemchat.getBemChatInfo();
                console.log(`[BemChatService] Loaded ${instanceType} instance: name=${info.name}, port=${info.port}, path=${info.path}`);
                // Sincronizar com banco de dados SQLite (apenas para API)
                if (instanceType === "api" && Number(info.port) > 0) {
                    try {
                        const { getDatabase } = require("../../utils/instanceDatabase");
                        const db = getDatabase();
                        db.addInstance({
                            name: file,
                            port: Number(info.port),
                            type: "api",
                            path: instancePath,
                        });
                        console.log(`[BemChatService] Synced ${file} to database (port ${info.port})`);
                    }
                    catch (error) {
                        console.warn(`[BemChatService] Could not sync ${file} to database:`, error);
                    }
                }
                this.bemchats.push(bemchat);
            }
            else {
                // Para plataforma: mantém lógica original
                const bemchat = new bemchat_plataform_1.BemChatPlataform({
                    name: file,
                    port: 0,
                    path: instancePath,
                });
                const info = bemchat.getBemChatInfo();
                console.log(`[BemChatService] Loaded ${instanceType} instance: name=${info.name}, port=${info.port}, path=${info.path}`);
                this.bemchats.push(bemchat);
            }
        }
        return foundNames;
    }
    listBemChats() {
        return this.bemchats.map((bemchat) => bemchat.getBemChatInfo());
    }
    async createBemChat(name, type, register, customPort) {
        // Verificar se já existe uma instância com esse nome E tipo
        const existingInstance = this.bemchats.find((bemchat) => {
            const info = bemchat.getBemChatInfo();
            const instanceType = bemchat instanceof bemchat_api_1.BemChatApi ? "api" : "plataform";
            return info.name === name && instanceType === type;
        });
        if (existingInstance) {
            return `Error: Instance ${name} already exists as a ${type} instance`;
        }
        // Verificar se a porta customizada já está em uso (se fornecida)
        if (customPort) {
            const portInUse = this.bemchats.find((bemchat) => {
                const info = bemchat.getBemChatInfo();
                // Plataforma usa 2 portas consecutivas (backend e frontend)
                if (type === "plataform") {
                    return info.port === customPort || info.port === customPort + 1;
                }
                else {
                    // API usa apenas 1 porta
                    return info.port === customPort;
                }
            });
            if (portInUse) {
                const portsInUse = type === "plataform"
                    ? `${customPort} or ${customPort + 1}`
                    : `${customPort}`;
                return `Error: Port ${portsInUse} is already in use by instance ${portInUse.getBemChatInfo().name}`;
            }
        }
        // Criar instância temporária para executar o método de criação
        const basePath = type === "api" ? this.apiPath : this.plataformPath;
        const instancePath = node_path_1.default.join(basePath, name);
        const tempInstance = type === "api"
            ? new bemchat_api_1.BemChatApi({ name, port: 0, path: instancePath })
            : new bemchat_plataform_1.BemChatPlataform({ name, port: 0, path: instancePath });
        // Executar método de criação
        if (!tempInstance.createBemChat) {
            return `Error: Create method not implemented for type ${type}`;
        }
        const result = await tempInstance.createBemChat(register, customPort);
        // Se criação foi bem-sucedida, recarregar lista de instâncias
        if (!result.startsWith("Error")) {
            this.loadBemChats();
        }
        return result;
    }
    async deleteBemChat(name) {
        const instance = this.bemchats.find((bemchat) => bemchat.getBemChatInfo().name === name);
        if (!instance)
            return `Error: Instance ${name} not found`;
        if (!instance.deleteBemChat) {
            return `Error: Delete method not implemented for this instance type`;
        }
        const result = await instance.deleteBemChat();
        // Se remoção foi bem-sucedida, recarregar lista de instâncias
        if (!result.startsWith("Error")) {
            this.loadBemChats();
        }
        return result;
    }
    async updateBemChat(name) {
        const instance = this.bemchats.find((bemchat) => bemchat.getBemChatInfo().name === name);
        if (!instance)
            return `bemchat ${name} not found`;
        return instance.updateBemChat();
    }
    async updateBemChats(instacesToUpdate = "all") {
        const apis = instacesToUpdate === "plataform"
            ? []
            : this.bemchats.filter((bemchat) => bemchat instanceof bemchat_api_1.BemChatApi);
        const plataforms = instacesToUpdate === "api"
            ? []
            : this.bemchats.filter((bemchat) => bemchat instanceof bemchat_plataform_1.BemChatPlataform);
        const results = [];
        for (const instance of [...apis, ...plataforms]) {
            results.push(await instance.updateBemChat());
        }
        if (results.length === 0)
            return "No bemchats updated";
        return results.join("\n");
    }
    async updateDatabase(name) {
        const sourceScriptPath = node_path_1.default.join(this.plataformPath, "update-db.sh");
        if (name === "all") {
            const plataforms = this.bemchats.filter((bemchat) => bemchat instanceof bemchat_plataform_1.BemChatPlataform);
            if (plataforms.length === 0)
                return "No plataform instances found to update database";
            const results = [];
            for (const instance of plataforms) {
                if (instance.updateDatabase) {
                    const result = await instance.updateDatabase(sourceScriptPath);
                    results.push(result);
                }
            }
            return results.join("\n");
        }
        const instance = this.bemchats.find((bemchat) => bemchat.getBemChatInfo().name === name);
        if (!instance)
            return `Instance ${name} not found`;
        if (instance instanceof bemchat_plataform_1.BemChatPlataform && instance.updateDatabase) {
            return await instance.updateDatabase(sourceScriptPath);
        }
        return `Database update not supported for instance ${name}`;
    }
    async restartInstance(name, type) {
        // Create temporary instance following the same pattern as createBemChat
        const basePath = type === "api" ? this.apiPath : this.plataformPath;
        const instancePath = node_path_1.default.join(basePath, name);
        // Verify instance path exists
        if (!node_fs_1.default.existsSync(instancePath)) {
            return `Error: ${type} instance ${name} not found at ${instancePath}`;
        }
        // Create temporary instance to execute restart
        const tempInstance = type === "api"
            ? new bemchat_api_1.BemChatApi({ name, port: 0, path: instancePath })
            : new bemchat_plataform_1.BemChatPlataform({ name, port: 0, path: instancePath });
        // Check if restart method exists
        if (!tempInstance.restart) {
            return `Error: Restart method not implemented for type ${type}`;
        }
        // Execute restart
        return await tempInstance.restart();
    }
    doNothing() {
        console.log("doing nothing");
    }
}
exports.BemChatService = BemChatService;
const bemChatService = new BemChatService();
exports.bemChatService = bemChatService;
