"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.getDatabase = getDatabase;
const better_sqlite3_1 = __importDefault(require("better-sqlite3"));
const node_path_1 = __importDefault(require("node:path"));
/**
 * Classe para gerenciar o banco de dados SQLite de instâncias
 */
class InstanceDatabase {
    db;
    constructor(dbPath) {
        const defaultPath = node_path_1.default.join(process.env.API_INSTANCES_PATH || "/srv/bemchat/api", "instances.db");
        this.db = new better_sqlite3_1.default(dbPath || defaultPath);
        this.initialize();
    }
    /**
     * Inicializa o banco de dados criando a tabela se não existir
     */
    initialize() {
        this.db.exec(`
      CREATE TABLE IF NOT EXISTS instances (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL UNIQUE,
        port INTEGER NOT NULL UNIQUE,
        type TEXT NOT NULL CHECK(type IN ('api', 'plataform')),
        path TEXT NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP
      );
      
      CREATE INDEX IF NOT EXISTS idx_port ON instances(port);
      CREATE INDEX IF NOT EXISTS idx_name ON instances(name);
    `);
    }
    /**
     * Adiciona uma nova instância ao banco de dados
     */
    addInstance(instance) {
        const stmt = this.db.prepare(`
      INSERT INTO instances (name, port, type, path)
      VALUES (?, ?, ?, ?)
    `);
        try {
            stmt.run(instance.name, instance.port, instance.type, instance.path);
            console.log(`[InstanceDB] Added instance: ${instance.name} on port ${instance.port}`);
        }
        catch (error) {
            if (error instanceof Error && error.message.includes("UNIQUE")) {
                console.warn(`[InstanceDB] Instance ${instance.name} already exists, updating...`);
                this.updateInstance(instance);
            }
            else {
                throw error;
            }
        }
    }
    /**
     * Atualiza uma instância existente
     */
    updateInstance(instance) {
        const stmt = this.db.prepare(`
      UPDATE instances 
      SET port = ?, type = ?, path = ?
      WHERE name = ?
    `);
        stmt.run(instance.port, instance.type, instance.path, instance.name);
        console.log(`[InstanceDB] Updated instance: ${instance.name}`);
    }
    /**
     * Remove uma instância do banco de dados
     */
    removeInstance(name) {
        const stmt = this.db.prepare("DELETE FROM instances WHERE name = ?");
        const result = stmt.run(name);
        if (result.changes > 0) {
            console.log(`[InstanceDB] Removed instance: ${name}`);
        }
        else {
            console.warn(`[InstanceDB] Instance ${name} not found in database`);
        }
    }
    /**
     * Busca uma instância pelo nome
     */
    getInstance(name) {
        const stmt = this.db.prepare("SELECT * FROM instances WHERE name = ?");
        return stmt.get(name);
    }
    /**
     * Lista todas as instâncias
     */
    getAllInstances() {
        const stmt = this.db.prepare("SELECT * FROM instances ORDER BY port");
        return stmt.all();
    }
    /**
     * Busca todas as portas em uso
     */
    getUsedPorts() {
        const stmt = this.db.prepare("SELECT port FROM instances ORDER BY port");
        const rows = stmt.all();
        return rows.map((row) => row.port);
    }
    /**
     * Busca a primeira porta disponível a partir de uma porta inicial
     */
    getNextAvailablePort(startPort = 2000) {
        const usedPorts = this.getUsedPorts();
        // Se não há portas em uso, retornar a porta inicial
        if (usedPorts.length === 0) {
            return startPort;
        }
        // Procurar primeira porta livre
        let port = startPort;
        for (const usedPort of usedPorts) {
            if (port < usedPort) {
                // Encontrou um gap
                return port;
            }
            if (port === usedPort) {
                // Porta em uso, tentar a próxima
                port++;
            }
        }
        // Se chegou aqui, todas as portas até a última estão em uso
        // Retornar a próxima após a última porta usada
        return port;
    }
    /**
     * Verifica se uma porta está em uso
     */
    isPortInUse(port) {
        const stmt = this.db.prepare("SELECT COUNT(*) as count FROM instances WHERE port = ?");
        const result = stmt.get(port);
        return result.count > 0;
    }
    /**
     * Fecha a conexão com o banco de dados
     */
    close() {
        this.db.close();
    }
}
// Singleton instance
let dbInstance = null;
/**
 * Obtém a instância singleton do banco de dados
 */
function getDatabase() {
    if (!dbInstance) {
        dbInstance = new InstanceDatabase();
    }
    return dbInstance;
}
exports.default = InstanceDatabase;
